<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Transaction;
use Mail;
use Hash;
use App\Mail\NotifyMail;
use App\Mail\LoanMail;
use App\Mail\TransMail;
use App\Mail\RequestMail;
use Carbon\Carbon;

class UserController extends Controller
{
    public function index(Request $request)
    {   
        $title = "User Dashboard  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $ip = $this->getIp();
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->limit(5)->get();
        return view('dashboard.index', compact('title','user','trans','ip'));
    }

    public function transactions(Request $request)
    {   
        $title = "User Investment  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->paginate(20);
               
        return view('dashboard.transactions', compact('title','user','trans'));
    }

    public function local_transfer(Request $request)
    {   
        $title = "User Investment  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $balance = $user->balance;
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->paginate(20);
        
        if ($request->isMethod('GET')) {
            return view('dashboard.local-transfer', compact('title','user','trans'));
        }

        if ($request->isMethod('POST')) {
            $request->validate([
                'acc_number' => 'required|numeric|min:8',
                'amount' => 'required|numeric',
                'trans_des' => 'required',
            ]);
            $amount = $request->amount;
            $acc_number = $request->acc_number;
            $trans_des = $request->trans_des;
            $trans_type = "local";
            if($amount > $balance){
                return redirect("user/local-transfer")->with('status',['text'=>'Sorry, You do not have sufficient Balance for this Transaction','type'=>'danger']);
            }

            $request->session()->put('amount', $amount);
            $request->session()->put('acc_number', $acc_number);
            $request->session()->put('trans_des', $trans_des);
            $request->session()->put('trans_type', $trans_type);

            return redirect("/user/confirm-transfer");
        }
        
    }

    public function wire_transfer(Request $request)
    {   
        $title = "User Investment  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $balance = $user->balance;
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->paginate(20);
        
        if ($request->isMethod('GET')) {
            return view('dashboard.wire-transfer', compact('title','user','trans'));
        }

        if ($request->isMethod('POST')) {
            $request->validate([
                'acc_number' => 'required|numeric|min:8',
                'acc_name' => 'required|min:6',
                'amount' => 'required|numeric',
                'swift_code' => 'required|min:4',
                'iban' => 'required|min:4',
                'bank_name' => 'required|min:4',
                'country' => 'required',
                'trans_des' => 'required',
            ]);
            $amount = $request->amount;
            $acc_number = $request->acc_number;
            $acc_name = $request->acc_name;
            $trans_des = $request->trans_des;
            $swift_code = $request->swift_code;
            $iban = $request->iban;
            $bank_name = $request->bank_name;
            $country = $request->country;
            $trans_type = "wire";
            if($amount > $balance){
                return redirect("user/wire-transfer")->with('status',['text'=>'Sorry, You do not have sufficient Balance for this Transaction','type'=>'danger']);
            }

            $request->session()->put('amount', $amount);
            $request->session()->put('acc_number', $acc_number);
            $request->session()->put('trans_des', $trans_des);
            $request->session()->put('trans_type', $trans_type);
            $request->session()->put('acc_name', $acc_name);
            $request->session()->put('swift_code', $swift_code);
            $request->session()->put('iban', $iban);
            $request->session()->put('bank_name', $bank_name);
            $request->session()->put('country', $country);

            return redirect("/user/confirm-transfer");
        }
        
    }

    public function confirm_transfer(Request $request)
    {   
        $title = "User Investment  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->paginate(20);
               
        return view('dashboard.confirm-transfer', compact('title','user','trans'));
    }

    public function send_otp(Request $request)
    {   
        $title = "User Investment  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $otp = rand(111111,999999);
        $email = $user->email;
        $name = $user->first_name;

        DB::table('users')
            ->where('user_id', $user_id)
            ->update([
                'otp'=> $otp,
            ]);

        $details = [
            'user_id' => $user_id,
            'otp' => $otp,
            'name' => $name,
            'ip' => $this->getIp(),
        ];
               
        try {
            Mail::to($email)->send(new NotifyMail($details));
            return redirect("/user/confirm-otp")->with('status', ['text'=>'Check your email for OTP to complete Transaction','type'=>'success']);
        } catch (Throwable $e) {
             return redirect("/user/confirm-transfer")->with('status', ['text'=>'Error!, OTP could not be sent, please try again or contact admin','type'=>'danger']);
        }

    }

    public function confirm_otp(Request $request)
    {   
        $title = "User Investment  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $balance = $user->balance;
        $email = $user->email;
        $name = $user->first_name;
        $owner_acc_number = $user->acc_number;
        $currency = $user->currency;
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->paginate(20);
        
        if ($request->isMethod('GET')) {
            return view('dashboard.confirm-otp', compact('title','user','trans'));
        }

        if ($request->isMethod('POST')) {
            $request->validate([
                'otp' => 'required|numeric|min:4',
            ]);
            $otp = $request->otp;
            $login = User::where('otp', $otp)
                        ->where('user_id', $user_id)
                        ->first();

            if($login){
                $amount = $request->session()->get('amount');
                $acc_number = $request->session()->get('acc_number');
                $trans_des = $request->session()->get('trans_des');
                $acc_name = $request->session()->get('acc_name');
                $swift_code = $request->session()->get('swift_code');
                $iban = $request->session()->get('iban');
                $bank_name =$request->session()->get('bank_name');
                $country = $request->session()->get('country');
                $date = Carbon::now();
                $new_balance = $balance - $amount;

                $trans_id = rand(11111,99999);

                $details = [
                    'trans_id' => $trans_id,
                    'name' => $name,
                    'amount' => $amount,
                    'acc_number' => $owner_acc_number,
                    'date' => $date,
                    'balance' => $new_balance,
                    'currency' => $currency,
                ];
                if($request->session()->get('trans_type') == "local"){
                    Transaction::create([
                        'trans_type'=> "Debit",
                        'amount'=> $amount,
                        'acc_number'=> $acc_number,
                        'dot'=> Carbon::now(),
                        'trans_des'=> $trans_des,
                        'trans_id'=> $trans_id,
                        'user_id'=> $user_id,
                        'trans_status'=> "Pending",
                    ]);
                }else{
                    Transaction::create([
                        'trans_type'=> "Debit",
                        'amount'=> $amount,
                        'acc_name'=> $acc_name,
                        'acc_number'=> $acc_number,
                        'bank_name'=> $bank_name,
                        'swift_code'=> $swift_code,
                        'iban'=> $iban,
                        'country'=> $country,
                        'dot'=> Carbon::now(),
                        'trans_des'=> $trans_des,
                        'trans_id'=> $trans_id,
                        'user_id'=> $user_id,
                        'trans_status'=> "Pending",
                    ]);
                }

                $user = DB::table('users')
                    ->where('user_id', $user_id)
                    ->update([
                        'balance'=> $new_balance,
                    ]);
                
                    Mail::to($email)->send(new TransMail($details));
                return redirect("user/transfer-status")->with('status',['text'=>'Transaction Completed','type'=>'success']);
            }else{
                return redirect("user/confirm-transfer")->with('status',['text'=>'Error!, OTP does not match','type'=>'danger']);
            }

        }
        
    }

    public function transfer_status(Request $request)
    {   
        $title = "User Investment  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->paginate(20);
               
        return view('dashboard.transfer-status', compact('title','user','trans'));
    }


    public function payments(Request $request)
    {   
        $title = "Make Payments  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $balance = $user->balance;
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->paginate(20);
        
        if ($request->isMethod('GET')) {
            return view('dashboard.payments', compact('title','user','trans'));
        }

        if ($request->isMethod('POST')) {
            $request->validate([
                'category' => 'required',
                'amount' => 'required|numeric',
                'biller' => 'required',
                'transfer_type' => 'required',
            ]);
            $amount = $request->amount;
  
            if($amount > $balance){
                return redirect("user/payments")->with('status',['text'=>'Sorry, You do not have sufficient Balance for this Transaction','type'=>'danger']);
            }

            $details = [
                'email' => $user->email,
                'category' => $request->category,
                'amount' => $request->amount,
                'biller' => $request->biller,
                'transfer_type' => $request->transfer_type,
            ];

             $admin_email = config('global.admin_email');

            try {
                Mail::to($admin_email)->send(new RequestMail($details));
                return redirect("/user/payments")->with('status', ['text'=>'Your Payment is Being Processed','type'=>'success']);
            } catch (Throwable $e) {
                 return redirect("/user/payments")->with('status', ['text'=>'Error!, Payment could not be Processed, please try again or contact admin','type'=>'danger']);
            }

        }
        
    }

    public function loan(Request $request)
    {   
        $title = "Loan Application  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $balance = $user->balance;
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->paginate(20);
        
        if ($request->isMethod('GET')) {
            return view('dashboard.loan', compact('title','user','trans'));
        }

        if ($request->isMethod('POST')) {
            $request->validate([
                'loan_type' => 'required',
                'amount' => 'required|numeric',
                'tenure' => 'required',
                'payment_plan' => 'required',
                'reason' => 'required',
            ]);
            

            $details = [
                'email' => $user->email,
                'loan_type' => $request->loan_type,
                'amount' => $request->amount,
                'tenure' => $request->tenure,
                'payment_plan' => $request->payment_plan,
                'reason' => $request->reason,
            ];

             $admin_email = config('global.admin_email');
             
            try {
                Mail::to($admin_email)->send(new LoanMail($details));
                return redirect("/user/loan")->with('status', ['text'=>'Your Loan Request is Being Processed','type'=>'success']);
            } catch (Throwable $e) {
                 return redirect("/user/loan")->with('status', ['text'=>'Error!, Loan Request could not be Processed, please try again or contact admin','type'=>'danger']);
            }

        }
        
    }

    public function profile(Request $request)
    {   
        $title = "User Profile " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        
        if ($request->isMethod('GET')) {
            return view('dashboard.profile', compact('title','user'));
        }

         if ($request->isMethod('POST')) {

                $user = DB::table('users')
                    ->where('user_id', $user_id)
                    ->update([
                        'first_name'=> $request->input('first_name'),
                        'last_name'=> $request->input('last_name'),
                        'phone'=> $request->input('phone'),
                        'dob'=> $request->input('dob'),
                        'gender'=> $request->input('gender'),
                        'address'=> $request->input('address'),
                        'city'=> $request->input('city'),
                        'country'=> $request->input('country'),
                    ]);
             
            
            return redirect("user/profile")->with('status', ['text'=>'Your Profile has been updated','type'=>'success']);

        }
    }


    public function change_password(Request $request)
    {   
        $title = "Change Password " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        
        if ($request->isMethod('GET')) {
            return view('dashboard.change-password', compact('title','user'));
        }

         if ($request->isMethod('POST')) {

            $request->validate([
                'old_password' => 'required',
                'password' => 'required|min:6|confirmed',
            ]);

            $password = $request->old_password;


            if (Hash::check($password, $user->password)) {
             DB::table('users')
                ->where('user_id', $user_id)
                ->update([
                    'password'=> Hash::make($request->input('password')),

                ]);
             
            $request->session()->forget('user_id');
            return redirect("/login")->with('status', ['text'=>'Password has been updated','type'=>'success']);
        }else{

            return redirect("user/change-password")->with('status', ['text'=>'Sorry! Password does not match','type'=>'danger']);
        }

        }
    }

    public function profile_picture(Request $request)
    {   
        $title = "Change Password " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();

         if ($request->isMethod('POST')) {

            $request->validate([
                'profile_picture' => 'required|image|mimes:jpg,png,jpeg,gif|max:2048',
            ]);

            $image = $request->file('profile_picture');
            $imageName = time().'.'.$image->extension();
            $imageName = str_replace(' ', '-', $imageName);
            $request->file('profile_picture')->move('uploads', $imageName);

             DB::table('users')
                ->where('user_id', $user_id)
                ->update([
                    'profile_picture'=> $imageName,

                ]);
             

            return redirect("user/profile")->with('profile_status', ['text'=>'Successfully update profile Picture','type'=>'success']);
        

        }
    }

    public function feedback(Request $request)
    {   
        $title = "Customer Feedback  " . config('global.site_title');
        $user_id = $request->session()->get('user_id');
        $user = User::where('user_id', $user_id)->first();
        $balance = $user->balance;
        $trans = Transaction::orderBy('created_at', 'desc')->where('user_id', $user_id)->paginate(20);
        
        if ($request->isMethod('GET')) {
            return view('dashboard.feedback', compact('title','user','trans'));
        }

        if ($request->isMethod('POST')) {
            $request->validate([
                'message' => 'required',
            ]);
            

            $details = [
                'email' => $user->email,
                'message' => $request->message,
            ];

             return redirect("/user/feedback")->with('status', ['text'=>'Your feedback has been received, we will get back to you shortly.','type'=>'success']);
             
        }
        
    }

     public function logout(Request $request)
    {   
        $request->session()->forget('user_id');
        return redirect("/")->with('status', ['text'=>'Logged out Successfully','type'=>'success']);
    }

    public function site_mode(Request $request, $mode)
    {   

       if($mode== "dark-mode"){
            $request->session()->put('mode', "dark-mode");
        }else{
            $request->session()->put('mode', "bg-lighter");
        }

        $site_mode = $request->session()->get('mode');
        return redirect("/user/index");
        
    }

    public function getIp(){
        if(!empty($_SERVER['HTTP_CLIENT_IP'])){
            //ip from share internet
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        }elseif(!empty($_SERVER['HTTP_X_FORWARDED_FOR'])){
            //ip pass from proxy
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        }else{
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        return $ip;
    }
}
